%%%% A Matlab Code for 2D Non-Penalization SEMDOT Algorithm %%%%
% Example SEMDOT2D(200,200,0.5,3.5)
function SEMDOT2D(nelx,nely,vol,rmin)
%% INITIALIZATION
vx=repmat(vol,nely,nelx); vxPhys=vx;
Emin=0.001; maxloop=5000; nG=10; ngrid=nG-1; tolx=0.001; E0=1;
%% INITIALIZE MMA OPTIMIZER
nele=nely*nelx; m=1; n=nely*nelx; nelm=nely*nelx;
vxmin=1e-3*ones(nelm,1); vxmax=ones(nelm,1);
vxold1=vx(:); vxold2=vxold1;
a0=1; ai=0; ci=1000; di=0; low=ones(nelm,1); upp=1;
move=0.01; scale_obj=0.1;
%% FILTER
[H,Hs]=HHs(nelx,nely,rmin);
[Hn,Hns]=HnHns(nelx,nely);
%% INITIALIZE HEAVISIDE SMOOTH FUNCTION
beta=0.5; ER=0.5;
%% PREPARE FINITE ELEMENT ANALYSIS
dx=1; dy=1; thickness=1; kxy=100; KE=lk(kxy,dx,dy,thickness);
nodenrs=reshape(1:(1+nelx)*(1+nely),1+nely,1+nelx);
edofVec=reshape(nodenrs(1:end-1,1:end-1)+1,nelx*nely,1);
edofMat=repmat(edofVec,1,4)+repmat([0 nely+[1 0] -1],nelx*nely,1);
iK=reshape(kron(edofMat,ones(4,1))',16*nelx*nely,1);
jK=reshape(kron(edofMat,ones(1,4))',16*nelx*nely,1);
%% ELEMENTAL NODES AND COORDINATES
[nodex,nodey]=meshgrid(0:nelx,0:nely);
[fnx,fny]=meshgrid(0:1/ngrid:nelx,0:1/ngrid:nely);
%% DEFINE LOADS AND SUPPORTS
amplitude=1e-2;
F=amplitude*ones((nelx+1)*(nely+1),1);
k=1; fixeddofs=[nely/2+1-k:nely/2+k]; alldofs=[1:(nely+1)*(nelx+1)];
freedofs=setdiff(alldofs,fixeddofs);
loop=0; change=1;
%% START ITERATION
while (change > tolx || tol>0.001) && loop < maxloop
    loop=loop+1;
    beta=beta+ER;
    %% FE-ANALYSIS
    sK=reshape(KE(:)*(vxPhys(:)'*E0+(1-vxPhys(:))'*(Emin*E0)),16*nelx*nely,1);
    K=sparse(iK,jK,sK); K=(K+K')/2;
    U=zeros((nelx+1)*(nely+1),1); U(freedofs)=K(freedofs,freedofs)\F(freedofs);
    %% OBJECTIVE FUNCTION AND SENSITIVITY ANALYSIS
    ce=reshape(sum((U(edofMat)*KE).*U(edofMat),2),nely,nelx);
    c(loop)=sum(sum((vxPhys.*E0+(1-vxPhys).*(Emin*E0)).*ce));
    dc=-((1-vxPhys)*Emin+vxPhys).*E0.*ce;
    dv=ones(nely,nelx);
    %% FILTERING
    dc(:)=H*(dc(:)./Hs);
    dv(:)=H*(dv(:)./Hs);
    %%  UPDATE DESIGN VARIABLES AND FILTERED ELEMENTAL VOLUME FRACTIONS
    vxval=vx(:); xmax=min(1,vxval+move); xmin=max(0.001,vxval-move);
    f0val=scale_obj*c; df0dx=scale_obj*dc(:);
    fval=sum(vxPhys(:))/(vol*nely*nelx)-1;
    dfdx=dv(:)'/(vol*nely*nelx);
    [vxmma,~,~,~,~,~,~,~,~,low,upp] = ...
        mmasub(m,n,loop,vxval,vxmin,vxmax,vxold1,vxold2,f0val,df0dx,fval,dfdx,low,upp,a0,ai,ci,di);
    vxnew=reshape(vxmma,nely,nelx); vxold2=vxold1(:); vxold1=vx(:);
    vxPhys(:)=(H*vxnew(:))./Hs;
    vxPhys=(vxPhys+flipud(vxPhys))/2;
    xn=reshape((Hn*vxPhys(:)./Hns),nely+1,nelx+1);
    %% UPDATE POINT DESNIGY BY A HEAVISIDE SMOOTH/STEP FUNCTION
    xg=interp2(nodex,nodey,xn,fnx,fny,'linear');
    l1=0; l2=1;
    while (l2-l1)>1e-5
        ls=(l1+l2)/2;
        xgnew=max(0.001,(tanh(beta*ls)+tanh(beta*(xg-ls)))/(tanh(beta*ls)+tanh(beta*(1-ls))));
        if sum(sum(xgnew))/((ngrid*nelx+1)*(ngrid*nely+1))-sum(vxPhys(:))/(nelx*nely)>0, l1=ls; else, l2=ls; end
    end
    %% ASSEMBLE GRID POINTS TO ELEMENTS
    vxPhys(:)=0;
    Terr=0; Tm=[];
    for i=1:nelx
        for j=1:nely
            e=(i-1)*nely+j;
            for i1=ngrid*(i-1)+1:ngrid*i+1
                for j1 = ngrid*(j-1)+1:ngrid*j+1
                    Tm = [Tm;xgnew(j1,i1)];
                    vxPhys(e)=vxPhys(e)+xgnew(j1,i1);
                end
            end
            if min(Tm)>0.001 && max(Tm)<1, Terr=Terr+1; end
            Tm=[];
        end
    end
    vxPhys=vxPhys/(ngrid+1)^2;
    %% CHECK CONVERGENCE
    change=sum(abs(vxnew(:)-vx(:)))/(vol*nely*nelx);
    tol=Terr/nele;
    vx=vxnew;
    %% PLOT RESULTS
    fprintf('It.:%3i Obj.:%8.4f Vol.:%4.3f ch.:%4.5f Topo.:%7.5f Beta.:%g.\n',loop,c(loop),mean(vxPhys(:)),change,tol,beta);
    contourf(fnx, flipud(fny), xg-ls, [0 0]); axis equal tight off; drawnow
    %% UPDATE HEAVISIDE REGULARIZATION PARAMETER
    beta=beta+ER;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This code was written by:                                               %
%                                                                         %
% Dr. Yun-Fei Fu                                                          %
% Department of Mechanical Engineering, University of Alberta, Canada     %
%                                                                         %
% Prof. Bernard Rolfe                                                     %
% School of Engineering, Faculty of Science Engineering and               %
% Built Environment, Deakin University, Australia                         %
%                                                                         %
% Please send your comments and questions to: jackyunfei.fu@outlook.com   %
%                                                                         %
% The code is intended for educational purposes and theoretical details   %
% are discussed in the book: "Smooth Topological Design of Continuum      %
% Structures", YF Fu and B Rolfe, CRC press.                              %
%                                                                         %
%                                                                         %
% The basic structure of the code is based on the 88-line code from:      %
% "Efficient topology optimization in MATLAB using 88 lines of code",     %
%  E Andreassen, A Clausen, M Schevenels, BS Lazarov and O Sigmund,       %
%  SMO 2010, doi:10.1007/s00158-010-0594-7                                %
%                                                                         %
% Disclaimer:                                                             %
% The authors do not guarantee that the code is free from errors.         %
% Furthermore, the authors shall not be liable in any event caused by the %
% use of the program.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


